/*
* A11y Reader (Hover + optional Autoplay)
* Aktivierung via <html data-a11y="a11y-text--read"> → startet Hover-Mode automatisch.
* Autoplay kann jederzeit manuell über die Konsole gestartet werden (A11yReader.startAutoplay()).
* NEU: Autoplay wird nur unterbrochen, wenn der Hover-Dwell erreicht ist
*      (also nicht bei jeder kleinen Mausbewegung).
* NEU: Beim Aktivieren gibt es eine Audio-Ansage mit kurzer Erklärung.
*/

(() => {
  if (!('speechSynthesis' in window)) return console.warn('Web Speech API nicht verfügbar.');

  // --- Einziges Token ---
  const TOKEN = 'a11y-text--read';

  // --- Zustand ---
  const STATE = {
    active: false,            // Feature an/aus
    mode: 'hover',            // 'hover' | 'autoplay'
    dwellMs: 400,             // Hover-Verweildauer (ms)
    dwellTimer: null,         // Timer-Handle
    lastHoverEl: null,        // zuletzt "behovertes" Element
    voice: null,              // gewählte TTS-Stimme
    rate: 1.10,               // Lesegeschwindigkeit
    autoplayList: [],         // Kandidatenliste
    autoplayIdx: -1,          // aktueller Index
    lastUtterance: null       // aktuell sprechende Utterance
  };

  // --- Logging-Helfer ---
  function logMode(reason) {
    const stateStr = STATE.active ? `mode=${STATE.mode}` : 'disabled';
    if (reason) {
      // console.log(`[A11yReader] ${stateStr} (${reason})`);
    } else {
      // console.log(`[A11yReader] ${stateStr}`);
    }
  }

  // --- Ansage beim Aktivieren ---
  function announceActivation() {
    const msg = "Website vorlesen aktiviert. Um die Website automatisch vorzulesen, nutze den Abspielen-Button in der unteren rechten Ecke der Seite. Oder bewege die Maus über einzelne Elemente, um nur dieses Element vorzulesen.";
    speak(msg);
  }

  // --- Öffentliche API (Konsole) ---
  window.A11yReader = {
    enable() {
      if (STATE.active) return;
      STATE.active = true;
      pickVoice();
      syncModeBehaviors();
      logMode('enable()');
      announceActivation();
    },
    disable() {
      if (!STATE.active) return;
      stopSpeech();
      STATE.active = false;
      clearDwell();
      stopAutoplay();
      logMode('disable()');
    },
    startHover() {
      if (STATE.mode !== 'hover') {
        STATE.mode = 'hover';
        syncModeBehaviors();
        logMode('console startHover');
      }
    },
    startAutoplay() {
      if (STATE.mode !== 'autoplay') {
        STATE.mode = 'autoplay';
        if (STATE.active) startAutoplay();
        else this.enable();
        logMode('console startAutoplay');
      }
    },
    next() { if (STATE.mode === 'autoplay') focusNextAutoplay(); },
    prev() { if (STATE.mode === 'autoplay') focusPrevAutoplay(); },
    setRate(rate) {
      const r = Number(rate);
      if (!Number.isNaN(r) && r > 0 && r <= 3) STATE.rate = r;
    },
    setDwell(ms) {
      const v = Number(ms);
      if (!Number.isNaN(v) && v >= 0) STATE.dwellMs = v;
    },
    status() {
      return { ...STATE, lastHoverEl: undefined, lastUtterance: undefined };
    }
  };

  // --- TTS ---
  function stopSpeech() {
    try { speechSynthesis.cancel(); } catch {}
    if (STATE.lastUtterance) {
      STATE.lastUtterance.onend = null;
      STATE.lastUtterance.onerror = null;
      STATE.lastUtterance = null;
    }
  }

  function pickVoice() {
    const voices = speechSynthesis.getVoices();
    if (!voices.length) return;

    const docLang = document.documentElement.lang || '';
    const userLang = navigator.language || '';
    const prefs = [docLang, userLang, 'de-DE','de-AT','de-CH','en-US','en-GB'].filter(Boolean);

    // 1. Stimmen mit „Premium“-Charakter (Google, Siri, …)
    const premiumMatch = voices.find(v =>
      /(Google|Siri|Neural|Premium|Enhanced)/i.test(v.name) &&
      prefs.some(l => (v.lang || '').startsWith(l))
    ) || voices.find(v =>
      /Anna/i.test(v.name) &&
      prefs.some(l => (v.lang || '').startsWith(l))
    );
    if (premiumMatch) {
      STATE.voice = premiumMatch;
      logMode(`picked premium voice: ${premiumMatch.name} (${premiumMatch.lang})`);
      return;
    }

    // 2. Irgendeine passende Sprache
    const langMatch = voices.find(v =>
      prefs.some(l => (v.lang || '').startsWith(l))
    );
    if (langMatch) {
      STATE.voice = langMatch;
      logMode(`picked lang voice: ${langMatch.name} (${langMatch.lang})`);
      return;
    }

    // 3. Irgendeine englische Stimme (Fallback)
    const enMatch = voices.find(v => (v.lang || '').startsWith('en-'));
    if (enMatch) {
      STATE.voice = enMatch;
      logMode(`picked en fallback: ${enMatch.name} (${enMatch.lang})`);
      return;
    }

    // 4. Letzter Fallback
    STATE.voice = voices[0];
    logMode(`picked default voice: ${STATE.voice?.name || 'none'}`);
  }
  if (speechSynthesis.onvoiceschanged !== undefined) {
    speechSynthesis.onvoiceschanged = () => { if (!STATE.voice) pickVoice(); };
  }

  function speak(text) {
    if (!text) return false;
    stopSpeech();
    const u = new SpeechSynthesisUtterance(text);
    if (STATE.voice) u.voice = STATE.voice;
    u.lang = STATE.voice?.lang || document.documentElement.lang || 'de-DE';
    u.rate = STATE.rate;

    if (STATE.mode === 'autoplay') {
      u.onend   = () => { if (STATE.active && STATE.mode === 'autoplay') focusNextAutoplay(); };
      u.onerror = () => { if (STATE.active && STATE.mode === 'autoplay') focusNextAutoplay(); };
    }

    STATE.lastUtterance = u;
    speechSynthesis.speak(u);
    return true;
  }

  // --- Zielauswahl ---
  const nativeSelectors = [
    GLOBALS.focusableElements
  ];

  const nonFocusableSelectors = [
    ':is(h1, h2, h3, h4, h5, h6, p, img[title], img[alt])',
    '.m-article :is(li)',
  ];

  const nonFocusableSelector = nonFocusableSelectors.join(', ');
  const focusableSelector    = [...nativeSelectors, ...nonFocusableSelectors].join(', ');

  // --- Sichtbarkeit / Blockaden ---
  function isBlocked(el) {
    return !!el.closest('[inert], [aria-hidden="true"], [hidden]');
  }

  function isVisible(el) {
    const s = window.getComputedStyle(el);
    if (s.visibility === 'hidden' || s.display === 'none') return false;
    const rect = el.getBoundingClientRect();
    return !(rect.width === 0 && rect.height === 0);
  }

  function isCandidate(el) {
    return el && el.nodeType === 1 && el.matches(focusableSelector) && isVisible(el) && !isBlocked(el);
  }

  function ensureProgrammaticFocus(el) {
    if (!el.matches(nonFocusableSelector)) return;
    if (!el.hasAttribute('tabindex')) el.setAttribute('tabindex', '-1');
  }

  // --- Text-Ermittlung ---
  function findLabelText(control) {
    if (control.id) {
      const lab = document.querySelector(`label[for="${CSS.escape(control.id)}"]`);
      if (lab) return lab.innerText.trim();
    }
    const parentLab = control.closest('label');
    if (parentLab) return parentLab.innerText.replace(control.value||'','').trim();
    return '';
  }

  function readableText(el) {
    if (!el) return '';
    const aria = el.getAttribute('aria-label');
    if (aria) return aria.trim();
    const labelledBy = el.getAttribute('aria-labelledby');
    if (labelledBy) {
      const txt = labelledBy.split(/\s+/).map(id => document.getElementById(id))
        .filter(Boolean).map(n => n.innerText || n.textContent || '')
        .join(' ').trim();
      if (txt) return txt;
    }
    if (el instanceof HTMLInputElement) {
      const type = (el.type||'').toLowerCase();
      const label = findLabelText(el);
      if (label) return label;
      if (el.placeholder) return el.placeholder;
      if (['submit','button','reset'].includes(type)) return el.value||el.name||'Schaltfläche';
      if (['checkbox','radio'].includes(type)) return (el.checked?'Ausgewählt: ':'')+(el.value||label||'Option');
      return el.value||el.name||el.id||'';
    }
    if (el instanceof HTMLTextAreaElement) return findLabelText(el) || el.placeholder || el.value || '';
    if (el instanceof HTMLSelectElement) {
      const label = findLabelText(el);
      const opt = el.selectedOptions?.[0]?.text || '';
      return [label,opt].filter(Boolean).join(': ');
    }
    if (el instanceof HTMLImageElement) return el.alt || el.title || 'Bild';
    if (el instanceof HTMLAnchorElement) return el.title || el.innerText.trim() || el.href || 'Link';
    if (el instanceof HTMLButtonElement) return el.title || el.innerText.trim() || el.value || 'Schaltfläche';
    const title = el.getAttribute('title'); if (title) return title.trim();
    const text = (el.innerText||el.textContent||'').replace(/\s+/g,' ').trim();
    if (text) return text;
    return el.getAttribute('role') || '';
  }

  // --- Fokus → Vorlesen ---
  function onFocusRead(e) {
    if (!STATE.active) return;
    const el = e.target;
    if (!isCandidate(el)) return;
    const txt = readableText(el);
    if (txt) speak(txt);
  }
  document.addEventListener('focusin', onFocusRead, true);

  // --- Hover-Mode ---
  function clearDwell() {
    if (STATE.dwellTimer) { clearTimeout(STATE.dwellTimer); STATE.dwellTimer = null; }
    STATE.lastHoverEl = null;
  }

  function focusViaHover(el) {
    if (!el) return;
    if (el.matches(nonFocusableSelector)) ensureProgrammaticFocus(el);
    requestAnimationFrame(() => {
      try { el.focus({ preventScroll: false }); } catch { el.focus(); }
    });
  }

  function scheduleDwell(el) {
    clearDwell();
    if (!isCandidate(el)) return;
    STATE.lastHoverEl = el;

    const dwellTime = STATE.mode === 'autoplay' ? STATE.dwellMs * 4 : STATE.dwellMs;

    STATE.dwellTimer = setTimeout(() => {
      if (!STATE.active) return;

      if (STATE.mode === 'autoplay') {
        stopAutoplay();
        STATE.mode = 'hover';
        logMode('dwell reached → abort autoplay → hover');
      }

      if (STATE.mode === 'hover' && STATE.lastHoverEl === el) {
        focusViaHover(el);
      }
    }, dwellTime);
  }

  document.addEventListener('mousemove', (e) => {
    if (!STATE.active) return;
    const target = /** @type {Element} */ (e.target);
    const el = target && (target.closest ? target.closest(focusableSelector) : null);
    if (!el) { clearDwell(); return; }
    if (STATE.lastHoverEl !== el) scheduleDwell(el);
  }, { passive: true });

  document.addEventListener('mouseleave', () => { if (STATE.mode === 'hover') clearDwell(); });
  document.addEventListener('mouseout', (e) => {
    if (STATE.mode !== 'hover') return;
    const to = e.relatedTarget, from = e.target;
    if (STATE.lastHoverEl && from === STATE.lastHoverEl && (!to || !STATE.lastHoverEl.contains(to))) clearDwell();
  });

  // --- Autoplay ---
  function buildAutoplayList() {
    const all = Array.from(document.querySelectorAll(focusableSelector));
    STATE.autoplayList = all.filter(isCandidate);
  }

  function focusElement(el, opts = { scroll: true }) {
    if (!el) return;
    if (el.matches(nonFocusableSelector)) ensureProgrammaticFocus(el);
    try {
      if (opts.scroll) {
        try { el.scrollIntoView({ block: 'center', inline: 'nearest', behavior: 'auto' }); } catch {}
      }
      el.focus({ preventScroll: !opts.scroll });
    } catch { el.focus(); }
  }

  function focusNextAutoplay() {
    if (!STATE.active || STATE.mode !== 'autoplay') return;
    let i = STATE.autoplayIdx + 1;
    while (i < STATE.autoplayList.length) {
      const el = STATE.autoplayList[i];
      if (isCandidate(el)) {
        STATE.autoplayIdx = i;
        focusElement(el, { scroll: true });

        requestAnimationFrame(() => {
          if (!STATE.active || STATE.mode !== 'autoplay') return;
          const txt = readableText(el);
          if (!txt) { setTimeout(focusNextAutoplay, 0); return; }
          if (document.activeElement !== el) speak(txt);
        });

        return;
      }
      i++;
    }
    stopAutoplay();
    logMode('autoplay finished');
  }

  function focusPrevAutoplay() {
    if (!STATE.active || STATE.mode !== 'autoplay') return;
    let i = STATE.autoplayIdx - 1;
    while (i >= 0) {
      const el = STATE.autoplayList[i];
      if (isCandidate(el)) {
        STATE.autoplayIdx = i;
        focusElement(el, { scroll: true });
        return;
      }
      i--;
    }
  }

  function startAutoplay() {
    stopSpeech();
    buildAutoplayList();
    STATE.autoplayIdx = -1;
    logMode(`autoplay start (candidates=${STATE.autoplayList.length})`);
    focusNextAutoplay();
  }

  function stopAutoplay() {
    stopSpeech();
  }

  function syncModeBehaviors() {
    if (!STATE.active) return;
    if (STATE.mode === 'hover') {
      stopAutoplay();
      clearDwell();
      logMode('switch → hover');
    } else if (STATE.mode === 'autoplay') {
      clearDwell();
      startAutoplay();
    }
  }

  // --- Steuerung über <html data-a11y="a11y-text--read"> ---
  const htmlEl = document.documentElement;

  function parseTokens(val) {
    if (!val) return [];
    return val.split(',').map(s => s.trim()).filter(Boolean);
  }
  function hasToken() {
    return parseTokens(htmlEl.getAttribute('data-a11y')).includes(TOKEN);
  }

  function syncFromAttr() {
    const shouldBeActive = hasToken();

    if (shouldBeActive && !STATE.active) {
      STATE.active = true;
      STATE.mode = 'hover';
      pickVoice();
      syncModeBehaviors();
      logMode('attr enabled');
      announceActivation();
    } else if (!shouldBeActive && STATE.active) {
      window.A11yReader.disable();
      logMode('attr disabled');
    }
  }

  // Initial & Observer
  syncFromAttr();
  new MutationObserver((muts) => {
    for (const m of muts) {
      if (m.type === 'attributes' && m.attributeName === 'data-a11y') {
        syncFromAttr();
      }
    }
  }).observe(htmlEl, { attributes: true, attributeFilter: ['data-a11y'] });

})();
