/*
* visibility detection feature
*    --isVisible, --isFadingIn, --isFadingOut
*    --invertOnScroll
*/

document.addEventListener("DOMContentLoaded", () => {
	const articles = Array.from(document.querySelectorAll(".m-article"));

	const observerOptions = {
		root: null,
		rootMargin: "0px",
		threshold: buildThresholdList()
	};

	const observer = new IntersectionObserver((entries) => {
		entries.forEach(entry => {
			const target = entry.target;
			const ratio = entry.intersectionRatio;

			// Reset everything first
			target.classList.remove("m-article--isVisible", "m-article--isFadingIn", "m-article--isFadingOut");

			if (ratio === 1) {
				// Vollständig im Viewport
				target.classList.add("m-article--isVisible");
			} else if (ratio > 0 && ratio < 1) {
				// Teilweise sichtbar
				const bounding = entry.boundingClientRect;
				const viewportHeight = window.innerHeight || document.documentElement.clientHeight;

				if (bounding.top >= 0 && bounding.top < viewportHeight) {
					// Betritt Viewport
					target.classList.add("m-article--isFadingIn");
				} else {
					// Verlässt Viewport
					target.classList.add("m-article--isFadingOut");
				}
			}
			// Sonst: vollständig außerhalb → keine Klasse
		});
	}, observerOptions);

	articles.forEach(article => observer.observe(article));

	function buildThresholdList() {
		let thresholds = [];
		let numSteps = 50;

		for (let i = 0; i <= numSteps; i++) {
			let ratio = i / numSteps;
			thresholds.push(ratio);
		}
		return thresholds;
	}
});

window.addEventListener('load', () => {
  const root = document.documentElement;

  if (root.getAttribute('data-theme')) return;

  const targets = document.querySelectorAll('.m-article--invertOnScroll');
  if (!targets.length) return;

  let activeCount = 0;

  const observer = new IntersectionObserver(
    (entries) => {
      entries.forEach(entry => {
        if (entry.isIntersecting) {
          activeCount++;
        } else {
          activeCount = Math.max(0, activeCount - 1);
        }
      });

      if (activeCount > 0) {
        root.setAttribute('data-theme', 'black');
      } else {
        root.setAttribute('data-theme', '');
      }
    },
    {
      root: null,
      rootMargin: '-47% 0px -47% 0px',
      threshold: 0,
    }
  );

  targets.forEach(target => observer.observe(target));
});
