<?php

declare(strict_types=1);

namespace App\Backend;

use App\Theme\DesignPlusVersion;
use Contao\BackendModule;
use Contao\CoreBundle\Csrf\ContaoCsrfTokenManager;
use Contao\Environment;
use Contao\Input;
use Contao\Message;
use Contao\System;
use Doctrine\DBAL\Connection;

class ThemeAssistant extends BackendModule
{
    /**
     * Name des Backend-Templates
     *
     * @var string
     */
    protected $strTemplate = 'be_theme_assistant';

    protected function compile(): void
    {
        // Contao-Systemmeldungen (Confirmation/Error/Info) ins Template geben
        $this->Template->messages = Message::generate();

        // Action-URL fürs Formular
        $this->Template->action = Environment::get('request');

        // Code-Version aus composer.json deines Themes
        $this->Template->codeVersion = DesignPlusVersion::get();

        // CSRF-Token für das Formular (Backend-Request-Schutz)
        /** @var ContaoCsrfTokenManager $csrf */
        $csrf = System::getContainer()->get('contao.csrf.token_manager');
        $this->Template->requestToken = $csrf->getDefaultTokenValue();

        // Formular-Submit prüfen
        if (Input::post('FORM_SUBMIT') === 'tl_theme_assistant') {
            $action = Input::post('update_action'); // 'modules' | 'layouts'

            try {
                /** @var Connection $connection */
                $connection = System::getContainer()->get('database_connection');
                $projectDir = System::getContainer()->getParameter('kernel.project_dir');

                $sqlDir     = $projectDir . '/files/theme/theme-design/sql';

                if ($action === 'modules') {
                    // Nur Module aktualisieren
                    $this->runSqlFile($sqlDir . '/modules_full.sql', $connection, 'Module');
                } elseif ($action === 'layouts') {
                    // Nur Seitenlayouts aktualisieren
                    $this->runSqlFile($sqlDir . '/layouts_full.sql', $connection, 'Seitenlayouts');
                } else {
                    Message::addError('Unbekannte Aktion beim Theme-Update. Bitte Seite neu laden und erneut versuchen.');
                }
            } catch (\Throwable $e) {
                Message::addError('Fehler beim Ausführen der SQL-Updates: ' . $e->getMessage());
            }

            // Nach POST immer Redirect (POST/Redirect/GET)
            $this->reload();
        }
    }

    /**
     * Führt alle Statements aus einer SQL-Datei aus.
     */
    private function runSqlFile(string $filePath, Connection $connection, string $label): void
    {
        if (!is_file($filePath)) {
            Message::addError(sprintf('%s-Update: SQL-Datei nicht gefunden (%s).', $label, $filePath));
            return;
        }

        $sql = file_get_contents($filePath);

        if ($sql === false || trim($sql) === '') {
            Message::addError(sprintf('%s-Update: SQL-Datei ist leer oder konnte nicht gelesen werden.', $label));
            return;
        }

        // Unsere Export-SQL-Datei setzt jedes Statement ans Ende mit ";\n\n".
        $rawStatements = preg_split('/;\s*\n\s*\n/', trim($sql)) ?: [];

        $executed = 0;

        foreach ($rawStatements as $rawStatement) {
            $rawStatement = trim($rawStatement);
            if ($rawStatement === '') {
                continue;
            }

            // Header-Kommentarblock (nur Kommentare, kein INSERT) überspringen
            if (!str_contains($rawStatement, 'INSERT INTO')) {
                continue;
            }

            // Kommentarzeilen (beginnend mit "--") innerhalb des Blocks entfernen
            $lines = array_map('trim', explode("\n", $rawStatement));
            $lines = array_filter($lines, static function (string $line): bool {
                return $line !== '' && !str_starts_with($line, '--');
            });

            if (empty($lines)) {
                continue;
            }

            $statement = implode("\n", $lines);
            $statement = trim($statement);

            if ($statement === '') {
                continue;
            }

            // Semikolon für den Executor anhängen
            $statement .= ';';

            $connection->executeStatement($statement);
            ++$executed;
        }

        if ($executed > 0) {
            Message::addConfirmation(sprintf('%s-Update: %d SQL-Statements erfolgreich ausgeführt.', $label, $executed));
        } else {
            Message::addInfo(sprintf('%s-Update: Keine ausführbaren SQL-Statements gefunden.', $label));
        }
    }

}
