<?php

declare(strict_types=1);

namespace App\Command;

use Contao\CoreBundle\Framework\ContaoFramework;
use Doctrine\DBAL\Connection;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

#[AsCommand(
    name: 'theme:export-sql',
    description: 'Exportiert tl_module und tl_layout als SQL-Upsert-Skripte für das Theme.'
)]
class ExportThemeSqlCommand extends Command
{
    public function __construct(
        private readonly ContaoFramework $framework,
        private readonly Connection $connection,
    ) {
        parent::__construct();
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        // Contao-Framework initialisieren (falls Models/Helfer genutzt würden)
        $this->framework->initialize();

        // Projektroot ermitteln – bei CLI-Aufruf ist getcwd() der Projektroot
        $projectDir = getcwd();

        // Zielverzeichnis für die SQL-Dateien
        $targetDir = $projectDir . '/files/theme/theme-design/sql';

        if (!is_dir($targetDir)) {
            if (!mkdir($targetDir, 0777, true) && !is_dir($targetDir)) {
                $output->writeln('<error>Verzeichnis files/theme/theme-design/sql konnte nicht erstellt werden.</error>');
                return Command::FAILURE;
            }
        }

        // 1) tl_module exportieren
        $modules = $this->connection->fetchAllAssociative('SELECT * FROM tl_module');

        if (!empty($modules)) {
            $moduleSql  = $this->buildUpsertSql('tl_module', $modules);
            $moduleFile = $targetDir . '/modules_full.sql';

            file_put_contents($moduleFile, $moduleSql);

            $output->writeln('<info>tl_module exportiert nach: files/theme/theme-design/sql/modules_full.sql</info>');
        } else {
            $output->writeln('<comment>tl_module enthält keine Datensätze – es wurde keine modules_full.sql erzeugt.</comment>');
        }

        // 2) tl_layout exportieren
        $layouts = $this->connection->fetchAllAssociative('SELECT * FROM tl_layout');

        if (!empty($layouts)) {
            $layoutSql  = $this->buildUpsertSql('tl_layout', $layouts);
            $layoutFile = $targetDir . '/layouts_full.sql';

            file_put_contents($layoutFile, $layoutSql);

            $output->writeln('<info>tl_layout exportiert nach: files/theme/theme-design/sql/layouts_full.sql</info>');
        } else {
            $output->writeln('<comment>tl_layout enthält keine Datensätze – es wurde keine layouts_full.sql erzeugt.</comment>');
        }

        $output->writeln('<info>Export abgeschlossen.</info>');

        return Command::SUCCESS;
    }

    /**
     * Baut ein SQL-File mit INSERT ... ON DUPLICATE KEY UPDATE
     * für alle übergebenen Zeilen einer Tabelle.
     *
     * Für bestimmte tl_module-IDs wird nur ein "create if not exist" erzeugt
     * (kein inhaltliches Überschreiben beim Update).
     *
     * @param string $table Tabellenname (z. B. tl_module)
     * @param array  $rows  Array von Assoc-Arrays (DB-Zeilen)
     */
    private function buildUpsertSql(string $table, array $rows): string
    {
        if (empty($rows)) {
            return '';
        }

        $sql  = "-- Auto-generiertes Upsert-SQL für {$table}\n";
        $sql .= "-- Generiert mit theme:export-sql\n\n";

        $columns = array_keys($rows[0]);
        $conn    = $this->connection;

        /*
         * IDs in tl_module, die nur "create if not exist" machen sollen
         * (nicht überschreiben, wenn bereits vorhanden):
         * - Logo (3)
         * - Theme Konfiguration (26)
         * - Offcanvas Content (27 + 36)
         * - Footer Content (35)
         */
        $protectedModuleIds = [3, 26, 27, 35, 36];

        foreach ($rows as $row) {
            // Spaltenliste
            $columnList = implode(', ', $columns);

            // Values-Teil
            $values = [];
            foreach ($columns as $col) {
                $value = $row[$col];

                if ($value === null) {
                    $values[] = 'NULL';
                } elseif (is_numeric($value)) {
                    // Zahlen direkt ausgeben, ohne Quotes
                    $values[] = (string) $value;
                } else {
                    // Strings sauber quoten
                    $values[] = $conn->quote((string) $value);
                }
            }

            $valuesList = implode(', ', $values);

            $sql .= "INSERT INTO {$table} ({$columnList})\n";
            $sql .= "VALUES ({$valuesList})\n";

            // Sonderbehandlung: bestimmte tl_module-IDs nur "create if not exist"
            if ($table === 'tl_module' && isset($row['id']) && in_array((int) $row['id'], $protectedModuleIds, true)) {
                // Kein echtes Update, nur ein no-op, damit ON DUPLICATE syntaktisch okay ist
                $sql .= "ON DUPLICATE KEY UPDATE\n";
                $sql .= "  id = id;\n\n";
                continue;
            }

            // Standardfall: volle Upsert-Logik
            $sql .= "ON DUPLICATE KEY UPDATE\n";

            $updates = [];
            foreach ($columns as $col) {
                // Primary Key (id) nicht updaten
                if ($col === 'id') {
                    continue;
                }

                $updates[] = "  {$col} = VALUES({$col})";
            }

            $sql .= implode(",\n", $updates) . ";\n\n";
        }

        return $sql;
    }
}
