<?php

declare(strict_types=1);

namespace App\Command;

use Contao\CoreBundle\Framework\ContaoFramework;
use Doctrine\DBAL\Connection;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Process\Process;
use ZipArchive;

#[AsCommand(
    name: 'theme:release',
    description: 'Erstellt eine neue Version (Version Bump, SQL-Export, DB-Backup, Packaging)'
)]
class ThemeReleaseCommand extends Command
{
    public function __construct(
        private readonly ContaoFramework $framework,
        private readonly Connection $connection,
    ) {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this->addArgument('version', InputArgument::REQUIRED, 'Neue Version (z. B. 5.7.0)');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->framework->initialize();

        $version    = $input->getArgument('version');
        $projectDir = getcwd();
        $filesystem = new Filesystem();

        // -----------------------------------------------------------
        // 1a) Version in composer.json setzen
        // -----------------------------------------------------------
        $composerPath = $projectDir . '/composer.json';

        if (!is_file($composerPath)) {
            $output->writeln('<error>composer.json nicht gefunden!</error>');
            return Command::FAILURE;
        }

        $composer = json_decode(file_get_contents($composerPath), true);
        $composer['version'] = $version;
        file_put_contents($composerPath, json_encode($composer, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        $output->writeln("<info>Version in composer.json auf {$version} gesetzt.</info>");

        // -----------------------------------------------------------
        // 1b) Version in config/config.yml setzen (Einrückung bleibt)
        // -----------------------------------------------------------
        $configPath = $projectDir . '/config/config.yml';
        if (is_file($configPath)) {
            $configContent = file_get_contents($configPath);

            $newConfigContent = preg_replace(
                '/^(\s*)badge_title:\s*[\'"]Design\+\sv([\d\.]+)[\'"]/m',
                '$1badge_title: \'Design+ v' . $version . '\'',
                $configContent
            );

            file_put_contents($configPath, $newConfigContent);
            $output->writeln("<info>Version in config.yml auf {$version} gesetzt.</info>");
        }

        // -----------------------------------------------------------
        // 2) SQL-Exports erzeugen
        // -----------------------------------------------------------
        $exportCmd = new Process(['php', 'vendor/bin/contao-console', 'theme:export-sql']);
        $exportCmd->run();

        if (!$exportCmd->isSuccessful()) {
            $output->writeln("<error>Fehler beim SQL-Export: {$exportCmd->getErrorOutput()}</error>");
            return Command::FAILURE;
        }

        $output->writeln('<info>SQL-Exports erfolgreich erzeugt.</info>');

        // -----------------------------------------------------------
        // 3) Backup erzeugen
        // -----------------------------------------------------------
        $backup = new Process(['php', 'vendor/bin/contao-console', 'contao:backup:create']);
        $backup->run();

        if (!$backup->isSuccessful()) {
            $output->writeln("<error>Fehler beim Backup: {$backup->getErrorOutput()}</error>");
            return Command::FAILURE;
        }

        $output->writeln('<info>Backup erfolgreich erstellt.</info>');

        // -----------------------------------------------------------
        // 4) theme.cto Ordner befüllen
        // -----------------------------------------------------------
        $distDir = $projectDir . '/_dist';
        $ctoDir  = $distDir . '/theme.cto';

        if (!is_dir($distDir)) {
            $filesystem->mkdir($distDir);
        }

        if (!is_dir($ctoDir)) {
            $filesystem->mkdir($ctoDir);
        }

        $output->writeln("<info>Ordner _dist/theme.cto bereit.</info>");

        $foldersToCopy = [
            'config',
            'contao',
            'files',
            'src',
            'templates',
            'var/backups',
        ];

        foreach ($foldersToCopy as $folder) {
            $source = $projectDir . '/' . $folder;
            $target = $ctoDir . '/' . $folder;

            if (!is_dir($source)) {
                $output->writeln("<comment>Übersprungen (nicht vorhanden): {$folder}</comment>");
                continue;
            }

            if (is_dir($target)) {
                $filesystem->remove($target);
            }

            $filesystem->mirror($source, $target, null, [
                'override' => true,
                'delete'   => true,
            ]);

            $output->writeln("<info>Kopiert: {$folder}</info>");
        }

        // -----------------------------------------------------------
        // >>> NEU – theme.xml & composer.json mit kopieren
        // -----------------------------------------------------------
        foreach (['theme.xml', 'composer.json'] as $fileName) {
            $src = $projectDir . '/' . $fileName;
            $dest = $ctoDir . '/' . $fileName;

            if (is_file($src)) {
                $filesystem->copy($src, $dest, true);
                $output->writeln("<info>Datei kopiert: {$fileName}</info>");
            } else {
                $output->writeln("<comment>Übersprungen (nicht vorhanden): {$fileName}</comment>");
            }
        }

        // -----------------------------------------------------------
        // 5) CTO ZIP erzeugen (temporär)
        // -----------------------------------------------------------
        $tmpZip = $distDir . "/theme-v{$version}.cto";

        $zip = new ZipArchive();
        if ($zip->open($tmpZip, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            $output->writeln("<error>Konnte ZIP nicht erzeugen: {$tmpZip}</error>");
            return Command::FAILURE;
        }

        $finder = new Finder();
        $finder->files()->in($ctoDir);

        foreach ($finder as $file) {
            $filePath = $file->getRealPath();
            $relative = ltrim(str_replace($ctoDir, '', $filePath), '/');
            $zip->addFile($filePath, $relative);
        }

        $zip->close();

        $output->writeln("<info>CTO ZIP erstellt: {$tmpZip}</info>");

        // -----------------------------------------------------------
        // >>> NEU – ZIP in /_dist/version.zip/ ablegen (Ordner!)
        // -----------------------------------------------------------
        $versionsDir = $distDir . '/version.zip';

        if (!is_dir($versionsDir)) {
            $filesystem->mkdir($versionsDir);
        }

        // alle .cto Dateien dort löschen
        foreach (glob($versionsDir . '/*.cto') as $oldCto) {
            $filesystem->remove($oldCto);
        }

        // neue Datei dorthin verschieben
        $finalPath = $versionsDir . "/theme-v{$version}.cto";
        $filesystem->rename($tmpZip, $finalPath, true);

        $output->writeln("<info>Release gespeichert unter: {$finalPath}</info>");

        // -----------------------------------------------------------
        // >>> NEU – zweite Datei: /_dist/latest.cto.zip
        // -----------------------------------------------------------
        $latestZip = $distDir . '/latest.cto.zip';

        // ggf. alte Datei löschen
        if (is_file($latestZip)) {
            $filesystem->remove($latestZip);
        }

        // Kopie des .cto Zips anlegen
        $filesystem->copy($finalPath, $latestZip, true);

        $output->writeln("<info>Zusätzlich erstellt: {$latestZip}</info>");

        // -----------------------------------------------------------
        // DONE
        // -----------------------------------------------------------
        $output->writeln("\n<fg=green>🎉 Theme-Release {$version} erfolgreich erstellt! (inkl. CTO & Versioning)</>");

        return Command::SUCCESS;
    }
}
